<?php
/*
Plugin Name: Header Countdown Timer
Description: A customizable countdown timer displayed above the site header with dynamic settings, fixed end date, enable/disable option, link option, and enhanced responsive UI.
Version: 1.3
Author: Grok
*/

// جلوگیری از دسترسی مستقیم
if (!defined('ABSPATH')) {
    exit;
}

// افزودن منوی تنظیمات در پیشخوان
function hct_add_admin_menu() {
    add_options_page('Header Countdown Settings', 'Header Countdown', 'manage_options', 'header_countdown', 'hct_options_page');
}
add_action('admin_menu', 'hct_add_admin_menu');

// ثبت تنظیمات
function hct_settings_init() {
    register_setting('hct_pluginPage', 'hct_settings');
    add_settings_section(
        'hct_pluginPage_section',
        __('Header Countdown Timer Settings', 'header-countdown'),
        null,
        'hct_pluginPage'
    );
    add_settings_field(
        'hct_enabled',
        __('Enable Countdown', 'header-countdown'),
        'hct_enabled_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_discount_text',
        __('Discount Text', 'header-countdown'),
        'hct_discount_text_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_end_date',
        __('End Date (YYYY-MM-DD HH:MM:SS)', 'header-countdown'),
        'hct_end_date_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_start_date',
        __('Start Date (YYYY-MM-DD HH:MM:SS)', 'header-countdown'),
        'hct_start_date_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_end_message',
        __('End Message', 'header-countdown'),
        'hct_end_message_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_link_url',
        __('Countdown Link URL', 'header-countdown'),
        'hct_link_url_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_font_size',
        __('Font Size (px)', 'header-countdown'),
        'hct_font_size_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_text_color',
        __('Text Color', 'header-countdown'),
        'hct_text_color_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
    add_settings_field(
        'hct_background_color',
        __('Background Color', 'header-countdown'),
        'hct_background_color_render',
        'hct_pluginPage',
        'hct_pluginPage_section'
    );
}
add_action('admin_init', 'hct_settings_init');

// رندر فیلدهای تنظیمات
function hct_enabled_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='checkbox' name='hct_settings[hct_enabled]' <?php checked(isset($options['hct_enabled']) && $options['hct_enabled'], 1); ?> value='1'>
    <p class='description'>برای نمایش شمارشگر در بالای هدر تیک بزنید.</p>
    <?php
}

function hct_discount_text_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='text' name='hct_settings[hct_discount_text]' value='<?php echo isset($options['hct_discount_text']) ? esc_attr($options['hct_discount_text']) : 'تخفیف تا پایان شهریور'; ?>' style='width: 300px;'>
    <p class='description'>متن تخفیف را وارد کنید (مثال: تخفیف تا پایان شهریور).</p>
    <?php
}

function hct_start_date_render() {
    $options = get_option('hct_settings');
    if (empty($options['hct_start_date'])) {
        $default_start_date = date('Y-m-d H:i:s');
        $options['hct_start_date'] = $default_start_date;
        update_option('hct_settings', $options);
    }
    ?>
    <input type='text' name='hct_settings[hct_start_date]' value='<?php echo isset($options['hct_start_date']) ? esc_attr($options['hct_start_date']) : ''; ?>' style='width: 300px;'>
    <p class='description'>فرمت: YYYY-MM-DD HH:MM:SS (مثال: 2025-08-16 21:56:00)</p>
    <?php
}

function hct_end_date_render() {
    $options = get_option('hct_settings');
    if (empty($options['hct_end_date'])) {
        $default_end_date = '2025-08-31 23:59:59';
        $options['hct_end_date'] = $default_end_date;
        update_option('hct_settings', $options);
    }
    ?>
    <input type='text' name='hct_settings[hct_end_date]' value='<?php echo isset($options['hct_end_date']) ? esc_attr($options['hct_end_date']) : ''; ?>' style='width: 300px;'>
    <p class='description'>فرمت: YYYY-MM-DD HH:MM:SS (مثال: 2025-08-31 23:59:59)</p>
    <?php
}

function hct_end_message_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='text' name='hct_settings[hct_end_message]' value='<?php echo isset($options['hct_end_message']) ? esc_attr($options['hct_end_message']) : 'تخفیف به پایان رسید!'; ?>' style='width: 300px;'>
    <?php
}

function hct_link_url_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='url' name='hct_settings[hct_link_url]' value='<?php echo isset($options['hct_link_url']) ? esc_attr($options['hct_link_url']) : ''; ?>' style='width: 300px;'>
    <p class='description'>آدرس URL برای کلیک روی کادر شمارشگر (مثال: https://example.com). خالی بگذارید اگر نیازی به لینک نیست.</p>
    <?php
}

function hct_font_size_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='number' name='hct_settings[hct_font_size]' value='<?php echo isset($options['hct_font_size']) ? esc_attr($options['hct_font_size']) : '18'; ?>' min='10'>
    <?php
}

function hct_text_color_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='text' name='hct_settings[hct_text_color]' value='<?php echo isset($options['hct_text_color']) ? esc_attr($options['hct_text_color']) : '#ffffff'; ?>' class='color-picker'>
    <script>
        jQuery(document).ready(function($){
            $('.color-picker').wpColorPicker();
        });
    </script>
    <?php
}

function hct_background_color_render() {
    $options = get_option('hct_settings');
    ?>
    <input type='text' name='hct_settings[hct_background_color]' value='<?php echo isset($options['hct_background_color']) ? esc_attr($options['hct_background_color']) : '#dc2626'; ?>' class='color-picker'>
    <script>
        jQuery(document).ready(function($){
            $('.color-picker').wpColorPicker();
        });
    </script>
    <?php
}

// صفحه تنظیمات
function hct_options_page() {
    ?>
    <div class='wrap'>
        <h2>Header Countdown Timer Settings</h2>
        <form action='options.php' method='post'>
            <?php
            settings_fields('hct_pluginPage');
            do_settings_sections('hct_pluginPage');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// افزودن Color Picker به پیشخوان
function hct_enqueue_color_picker($hook_suffix) {
    if ('settings_page_header_countdown' === $hook_suffix) {
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }
}
add_action('admin_enqueue_scripts', 'hct_enqueue_color_picker');

// افزودن شمارشگر به بالای هدر
function hct_add_countdown_to_header() {
    $options = get_option('hct_settings');
    if (!isset($options['hct_enabled']) || !$options['hct_enabled']) {
        return;
    }
    $start_date = isset($options['hct_start_date']) ? esc_attr($options['hct_start_date']) : date('Y-m-d H:i:s');
    $end_date = isset($options['hct_end_date']) ? esc_attr($options['hct_end_date']) : '2025-08-31 23:59:59';
    $end_message = isset($options['hct_end_message']) ? esc_html($options['hct_end_message']) : 'تخفیف به پایان رسید!';
    $discount_text = isset($options['hct_discount_text']) ? esc_html($options['hct_discount_text']) : 'تخفیف تا پایان شهریور';
    $link_url = isset($options['hct_link_url']) ? esc_url($options['hct_link_url']) : '';
    $font_size = isset($options['hct_font_size']) ? intval($options['hct_font_size']) : 18;
    $text_color = isset($options['hct_text_color']) ? esc_attr($options['hct_text_color']) : '#ffffff';
    $background_color = isset($options['hct_background_color']) ? esc_attr($options['hct_background_color']) : '#dc2626';
    ?>
    <div class="header-countdown-container">
        <?php if (!empty($link_url)) { ?>
            <a href="<?php echo $link_url; ?>" class="header-countdown-link">
        <?php } ?>
        <div class="header-countdown-inner">
            <span class="discount-text"><?php echo $discount_text; ?></span>
            <div class="countdown-items">
                <span class="countdown-item"><span id="days"></span> روز</span>
                <span class="countdown-item"><span id="hours"></span> ساعت</span>
                <span class="countdown-item"><span id="minutes"></span> دقیقه</span>
                <span class="countdown-item"><span id="seconds"></span> ثانیه</span>
            </div>
        </div>
        <?php if (!empty($link_url)) { ?>
            </a>
        <?php } ?>
    </div>
    <script>
        const hctStartDate = new Date('<?php echo $start_date; ?>');
        const hctEndDate = new Date('<?php echo $end_date; ?>');
        function updateHeaderCountdown() {
            const now = new Date();
            const timeDiff = hctEndDate - now;
            if (timeDiff <= 0) {
                document.querySelector('.header-countdown-inner').innerHTML = '<?php echo $end_message; ?>';
                <?php if (!empty($link_url)) { ?>
                    document.querySelector('.header-countdown-link').removeAttribute('href');
                <?php } ?>
                return;
            }
            const days = Math.floor(timeDiff / (1000 * 60 * 60 * 24));
            const hours = Math.floor((timeDiff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((timeDiff % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((timeDiff % (1000 * 60)) / 1000);
            document.getElementById('days').innerText = days;
            document.getElementById('hours').innerText = hours;
            document.getElementById('minutes').innerText = minutes;
            document.getElementById('seconds').innerText = seconds;
        }
        setInterval(updateHeaderCountdown, 1000);
        updateHeaderCountdown();
    </script>
    <style>
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.03); }
            100% { transform: scale(1); }
        }
        .header-countdown-container {
            font-family: 'Vazirmatn', Arial, sans-serif;
            text-align: center;
            direction: rtl;
            background: linear-gradient(90deg, <?php echo $background_color; ?>, <?php echo $background_color . 'cc'; ?>);
            color: <?php echo $text_color; ?>;
            padding: 12px 0;
            position: sticky;
            top: 0;
            z-index: 1000;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            animation: fadeIn 0.8s ease-in-out;
        }
        .header-countdown-link {
            text-decoration: none;
            color: inherit;
            display: block;
        }
        .header-countdown-link:hover {
            background: linear-gradient(90deg, <?php echo $background_color . 'e6'; ?>, <?php echo $background_color . 'b3'; ?>);
        }
        .header-countdown-inner {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            padding: 0 20px;
            max-width: 1200px;
            margin: 0 auto;
            font-size: <?php echo $font_size; ?>px;
        }
        .discount-text {
            font-weight: bold;
            background: rgba(255, 255, 255, 0.2);
            padding: 6px 12px;
            border-radius: 8px;
            animation: pulse 2s infinite;
        }
        .countdown-items {
            display: flex;
            gap: 10px;
        }
        .countdown-item {
            background: rgba(255, 255, 255, 0.15);
            padding: 6px 12px;
            border-radius: 6px;
            transition: transform 0.3s ease;
        }
        .countdown-item:hover {
            transform: translateY(-2px);
        }
        .countdown-item span {
            font-weight: bold;
        }
        @media (max-width: 768px) {
            .header-countdown-container {
                font-size: <?php echo max(14, $font_size - 4); ?>px;
                padding: 10px 0;
            }
            .header-countdown-inner {
                flex-wrap: wrap;
                gap: 10px;
                padding: 0 15px;
            }
            .countdown-items {
                gap: 8px;
            }
            .countdown-item {
                padding: 5px 10px;
            }
        }
        @media (max-width: 480px) {
            .header-countdown-container {
                font-size: <?php echo max(12, $font_size - 6); ?>px;
                padding: 8px 0;
            }
            .header-countdown-inner {
                flex-direction: column;
                gap: 8px;
                padding: 0 10px;
            }
            .countdown-items {
                flex-wrap: wrap;
                justify-content: center;
                gap: 6px;
            }
            .countdown-item {
                padding: 4px 8px;
            }
        }
    </style>
    <?php
}
add_action('wp_body_open', 'hct_add_countdown_to_header');
?>